;**********************************************************************************************
;                 UART.asm - UARTS for Z8 Microcontrollers
;                      Copyright (C) 2000 ZiLOG, Inc
;                           All rights reserved
;**********************************************************************************************
;	Assembler: ZDS V3.62
;	Best if viewed in Courier New 9 pt with 7 space tabs
;**********************************************************************************************
;
;	                     UARTS FOR Z8 MICROCONTROLLERS
;
;  This program will provide an interrupt driven UART for the Z8 family microcontrollers.
;  This file has two main parts. The first part supports the hardware UART function
;  in the MUZE family parts. If UART_TYPE is set to one in the UART.INC file then
;  the hardware UART firmware is used. The second part is for UART_TYPE equal to zero.
;  This is the software uart routine that can be used with any other Z8 that is supported
;  by the Extreme Exprezzway Z8 SDK PCB. NOTE: Make sure to change the jumper settings (R16-R19)
;  on the SDK PCB to select correct Hardware UART or software UART I/O pins.
;
;  The Hardware UART has been tested at 1200, 2400, 9600, 19200 and 38400 Baud using a 12mhz crystal
;  The Software UART has been tested at 1200, 2400, 9600 and 19200 Baud using a 12mhz crystal
;  The Software UART can only operate in HALF DUPLEX mode (ie: only Tx or Rx at a given time)
;
;  System Resources Used for Hardware UART (with emulator workaround):
;	174 Bytes ROM	
;	3   Bytes RAM
;	2   I/O pins 
;	1   C/T (T1)
;
;  System Resources Used for Hardware UART (without emulator workaround):
;	156 Bytes ROM	
;	3   Bytes RAM
;	2   I/O pins 
;
;  System Resources Used for Software UART:
;	183 Bytes ROM	(182 Bytes ROM if UART_SHARING selected) (148 bytes without sharing)
;	5   Bytes RAM
;	1   C/T (T1)
;	2   I/O pins (P20 & P31) Note: the input pin must have edge interrupt capability 
;
;
;          !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!		
;          !               DO NOT EDIT THIS SOURCE FILE.                !
;          ! ALL APPLICATION CUSTOMIZATION IS DONE IN THE UART.INC FILE !
;          !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
;
;  User Customization possible:
;	Baud rate selection - Literal BAUDRATE is set equal to 1200,2400,4800,9600, or 19200
;								(38400 possible with HW UART)
;
;  To use this module:
;	1.  Add the UART.ASM module to your project Source Files
;	2.  Add the UART.INC module to your project Dependencies
;	3.  Edit the UART.INC module to set the desired Baudrate and select HW or SW UART
;	4.  Add the following statement to any source files acessing the UART:
;			INCLUDE	"UART.INC"
;
;  Interfacing to UART.ASM
;	UART.ASM provides a byte wide interface, and has seven status bits that the application
;	can interrogate to determine activity.
;	
;	Data Interface:
;		TxDATA is the output interface
;		RxDATA is the input interface
;
;	Status Interface (all status bits are in the UART_FLAGS RAM location):
;		RxRDY		 D0=>1 = Done Receiving (data ready to be read)
;		TxRDY		 D1=>1 = Done Transmitting (next data ready to be sent)
;		RX_FLAG		 D2=>1 = Recieving (or transmitting if 0)
;		UART_IRQ	 D3=>1 Set when UART done Tx or Rx, reset by IRQ svc
;		UART_BUSY	 D4=>1 Set when UART is running (SW UART ONLY) 
;		UART_DLY	 D5=>1 Set when UART is stopping (resource sharing - SW UART ONLY) 
;		UART_ERROR	 D6=>1 Set if the (HW) UART (Rx) detects an error
;
;	The application should only read status bits RxRDY,TxRDY and RX_FLAG.
;	UART_IRQ is a Read/Write status bit.  The application can read it to determine if
;	the UART requires service, and set it to 0 when service is completed.
;	UART.ASM does NOT generate any application servicable interrupts.
;	UART_ERROR is a Read/Write status bit.  The application can read it to determine if
;	the Rx Data is valid, or if an error has occured.  If an error has occured, the RxData
;	returned from the UART will be FF.  The UART_ERROR bit will remain set until the application
;	clears it.
;	UART_BUSY is a read only status bit for the SW UART, which when set indicats that the SW UART is
;	in the process of serializing/deserializing a byte. 
;
;  Sample Usage of UART.ASM
;
;	During Application intitalization:
;		CALL	INIT_UART
;
;	When wishing to start a byte transmission:
;		TM	UART_FLAGS,#TxRDY	; See if Tx is Ready for more data
;		JR	z,do_something_else 	; If not yet ready, the app can do something else
;		LD	TxDATA,xxxxx      		; If ready, load the data byte
;		CALL	SEND_BYTE            	; And start the transmission
;
;	When Checking to see if a character has been transmitted, or received:
;		TM	UART_FLAGS,#UART_IRQ 	; See if the UART needs service
;		JR	z,do_something_else    	; If not, the app can do something else
;		AND	UART_FLAGS.#~UART_IRQ  	; If so, first reset the IRQ flag
;		TM	UART_FLAGS,#TxRDY     	; Then determine if Tx or Rx needs service
;		JR	NZ,service_Tx         	; and service accordingly
;		TM	UART_FLAGS,#RxRDY
;		JR	NZ,service_Rx
;
;
	    
;**********************************************************************************************
;
;  Revision History for UART.ASM:
;
; 	v1.00 - Production Release							MNG	11/12/00
;**********************************************************************************************

	TITLE	"UART for Z8"

	DEFINE REG_UART, SPACE=RFILE
	
	INCLUDE "UART.INC"

	GLOBALS	ON

;**********************************************************************************************
; UART Variables
;**********************************************************************************************
	SEGMENT REG_UART

RxDATA		DS	1
UART_FLAGS	DS	1
TxDATA		DS	1
 If UART_TYPE=0
RCV_DATA_BUFF	DS	1	; Receive Data Buffer is only needed for SW UART
BitCount	DS	1	; Bit counter is only needed for SW UART	
 ENDIF
;**********************************************************************************************
;	Software Uart Ports 
;**********************************************************************************************
RxD_Bit	EQU  	00000100B	; P32- Rx Pin
TxD_Bit	EQU  	00000010B	; P01- Tx Pin

	SEGMENT CODE

;**********************************************************************************************
;
;	UART_SVC
;
;  This routine is provided to present a uniform interface for all ZiLOG device drivers.  Some
;  drivers require periodic service (RF, PLC) by the application.
;
;  The UART should not require such service, however it has been found that the HW UART can 
;  sometimes 'lock up' with an overrun error.  As a recovery mechanism, the UART_SVC routine
;  will check for Rx errors on the HW UART, and if present will reset them.  The SW UART simply
;  returns from the service with no action taken.
;
;
;**********************************************************************************************
UART_SVC:
 IF UART_TYPE=1

	PUSH	RP			;Make sure register pointer is saved.
	SRP	#ASCI			;Select ASCI Bank
	TM	STAT,#(OE | PE | FE)	; Check for any Rx error conditions
	JR	Z,UART_SVC_DONE		; If an error occured - deal with it
	AND	CNTLA,#~RE			;Disable the Rx
	AND	CNTLA,#~EFR			;Clear the error flags in the HW
	OR	CNTLA,#RE			;Enable the Rx
UART_SVC_DONE	
	POP	RP							;Restore register pointer
 ENDIF

 	RET
;**********************************************************************************************



 IF UART_TYPE=1	;IF true then using Hardware Uart else software Uart is used
 			;  the HW UART code is the first half of this source file

;**********************************************************************************************
;
;	INIT_Hardware_UART
;
;  Switch Register pointer to point to Uart registers Bank A.
;	Set up Baud rate, 8bit, No Parity, 1Stop.
;	Receiver Interrupt Enabled.
;
;**********************************************************************************************

INIT_UART:
		PUSH	RP					;Make sure register pointer is saved.
		SRP	#ASCI					;Select ASCI Bank
;**********************************************************************************************
;	Set the Baud rate
;**********************************************************************************************
		LD	ASTL,#HW_BAUD			;Set up Baud rate timer				
		LD	ASTH,#%00

;**********************************************************************************************
;
;	ASCI Control Register B Set-up
;	CNTLB	Bank AH register 04H
;
;MPBT			EQU		10000000b	Multiprocessor Bit Transmitter
; 											0 - Transmit 0 in MPB (Reset State)
;											1 - Transmit 1 in MPB 
; MP			EQU		01000000b	Multiprocessor Mode
; 											0 - Multiprocessor mode Disabled (Reset State)
;											1 - Multiprocessor mode Enabled
; PR			EQU		00100000b	Prescale (Write Only)
; 											0 - BRG / by 10 (Reset State)
;											1 - BRG / by 30
; PRO			EQU		00010000b	Parity Even/Odd
; 											0 - Even Parity (Reset State)
;											1 - Odd Parity
; DR			EQU		00001000b	Divide Ratio
; 											0 - Divide by 16 (Reset State)
;											1 - Divide by 64
; SS2_0		EQU		00000111b	Clock Source and Speed Bits 
; 											210 
; SS_DIV_1								000 - Divide by 1
; SS_DIV_2								001 - Divide by 2
; SS_DIV_4								010 - Divide by 4
; SS_DIV_8								011 - Divide by 8
; SS_DIV_16								100 - Divide by 16
; SS_DIV_32								101 - Divide by 32
; SS_DIV_64								110 - Divide by 64
;									111 - Reserved (Reset State)
								

;**********************************************************************************************
		LD	CNTLB,#(SS_DIV_16)	;Transmit 0 in MPB,Multiprocessor disabled,BRG/10
						;Even Parity, Divide by 16, Div sel 16
;**********************************************************************************************
;
;	ASCI Extension Control register
;	ASEXT	Bank AH register 05H
;
; RX			EQU		10000000b	RX data state (Read Only)
; BRG			EQU		00001000b	Baud Rate Generator Mode
; 											0 - Use Speed bits in reg 4 (Reset State)
;											1 - Use ASTH or ASTL value
; RIS			EQU		00000100b	RX Interrupt on Start Bit
; 											0 - No IRQ on Start Bit (Reset State)
;											1 - IRQ3 on Start Bit
; BD			EQU		00000010b	Break Detect (Read Only)
; 											0 - Valid Data Byte
;											1 - Break Detected
; SB			EQU		00000001b	Send Break
; 											0 - Normal Operation (Reset State)
;											1 - Send Break
;**********************************************************************************************
		LD	ASEXT,#(BRG)		;Baud Rate Generator use ASTH,ASTL						
						;Normal Operation(no Break)
;**********************************************************************************************
;
;	ASCI Control Register A Set-up
;	CNTLA	Bank AH register 03H
;
; MPE			EQU		10000000b	Multiprocessor Enable Bit
; 												0 - Receive all Bytes  ( Reset State)
;												1 - Filter Bytes with MPB = 0
; RE			EQU		01000000b	Receiver Enable
;												0 - ASCI Receiver Disabled ( Reset State)
;												1 - ASCI Receiver Enabled (P30 = RX)
; TE			EQU		00100000b	Transmitter Enable
; 												0 - ASCI Transmitter Disabled
;												1 - ASCI Transmitter Enabled (P37 = TX)
; MPBR			EQU		00001000b	Multiprocessor Bit Received (Read Only)
; 												0 - Multiprocessor Bit Not Received
;												1 - Muttiprocessor Bit Received 
; EFR			EQU		00001000b	Error Flag Reset (Write Only)
; 												0 - Clear Error Latches
;											 	1 - No Effect
; MOD2			EQU		00000100b	MOD2 Select Number of Data bits
; 											 	0 - 7 Data Bits (Reset State)	
;												1 - 8 Data Bits
; MOD1			EQU		00000010b	MOD1 Parity Enable Control
; 												0 - No Parity (Reset State)
;												1 - Enable Parity 
; MOD0			EQU		00000001b	MOD0 Number of Stop Bits
; 												0 - 1 Stop Bit	(Reset State)
;												1 - 2 Stop Bits
;**********************************************************************************************
		LD	CNTLA,#(RE | TE | MOD2)	;Multiprocessor rec. all, Rx enable,Tx enable
							;Clear error Latches, 8 bits, no Parity, 1 stop
												
;**********************************************************************************************
;
;	ASCI Status register
;	STAT	Bank AH register 08H
;
;RDRNE		EQU		10000000b	Receive Data Register Not Entry (Read Only)
; 											0 - Receive FIFO Empty
;											1 - Receive FIFO contains 1 or more Bytes
; OE			EQU		01000000b	Overrun Error (Read Only)
; 											0 - Receive OK
;											1 - Next Byte is a FIFO overrun
; PE			EQU		00100000b	Parity Error (Read Only)
; 											0 - Parity OK
;											1 - Parity Error
; FE			EQU		00010000b	Framing Error (Read Only)
; 											0 - Receive OK
;											1 - Framing Error
; RIE			EQU		00001000b	Receiver Interrupt Enable
; 											0 - No IRQ on Receive (Reset State)
;											1 - IRQ3 on RDRNE going high or Start bit
; TDRE			EQU		00000010b	Transmit Data Register Empty (Read Only)
; 											0 - Transmitter Working
;											1 - Transmit Buffer Empty
; TIE			EQU		00000001b	Transmitter Interrupt Enable
; 											0 - No IRQ on transmit (Reset State)
;											1 - IRQ3 on TDRE going high
;**********************************************************************************************

		LD	STAT,#(RIE | TIE)

;;;; THERE APPEARS TO BE A PROBLEM WITH THE E13x WHEN BOTH TX & RX INTS ENABLED - NO RX INTS GET THROUGH!!
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
; !!!!! THE E136 EMULATOR HAS A BUG THAT PREVENT Tx INTERRUPTS FROM THE ASCI FROM OCCURRING  !!!!!!!
; !!!!! A WORKAROUND USED HERE IS TO USE T1 AS A BIT TIME COUNTER, LET IT BE SERVICED BY THE !!!!!!!
; !!!!! SAME SERVICE ROUTINE AS THE ASCI, AND USE IT TO TEST THE Tx STATUS FOR EMPTY.        !!!!!!!
; !!!!! ALL WORKAROUND CODE IS CONDITIONAL AND THE EMULATOR CONSTANT MUST BE SET TO 1 IN     !!!!!!!
; !!!!! UART.INC          8/20/00 MNG								  !!!!!!!
;													  !!!!!!!
;													  !!!!!!!
 IF EMULATOR=1			;								  !!!!!!!
					;								  !!!!!!!
		LD	STAT,#(RIE)	;IRQ3 on RDRNE (Rx byte in), IRQ3 on TDRE (Tx byte out)   !!!!!!!
 ENDIF					;								  !!!!!!!
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


; Found that Rx errors (OE) were not cleared by the above process
; In order for the Error Flag Reset command to work, the Rx must be disabled FIRST!
;
	AND	CNTLA,#~RE		;Disable the Rx
	AND	CNTLA,#~EFR		;Clear the error flags in the HW
	OR	CNTLA,#RE		;Enable the Rx

	POP	RP							;Restore register pointer

; The Tx is ready for data as soon as intialization is completed, so TxRDY gets set now
; The Rx is enabled as soon as intialization is completed, so any inbound messages will be rec'd.

	LD	UART_FLAGS,#00000110B	; Initial state of Flags:
					;	RxRDY     = 0 - There is no Rx data
					;	TxRDY     = 1 - Ready for Tx data
					;	RX_FLG    = 1 - We'll be set for Rx after init
					;	UART_IRQ  = 0 - There is no transfer completed
					;  UART_BUSY = 0 - The UART is not running (BUSY)
					;  UART_DLY  = 0 - The UART has not just ended an Rx

;**********************************************************************************************
; 	Enable RX   		EN_RCV is a GLOBAL ENTRY POINT
;**********************************************************************************************

	OR	UART_FLAGS,#RX_FLAG		;Since we are Rx mode, set the Rx_FLAG
	AND	UART_FLAGS,#~RxRDY    	;Clear the RxRDY flag (no data avail yet.)
	AND 	IRQ,#0F7H             	;Clear SIO interrupt.
	OR 	IMR,#00001000B		;Enable Hardware ASCI interrupts 

;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
; !!!!! THE E136 EMULATOR HAS A BUG THAT PREVENT Tx INTERRUPTS FROM THE ASCI FROM OCCURRING  !!!!!!!
; !!!!! A WORKAROUND USED HERE IS TO USE T1 AS A BIT TIME COUNTER, LET IT BE SERVICED BY THE !!!!!!!
; !!!!! SAME SERVICE ROUTINE AS THE ASCI, AND USE IT TO TEST THE Tx STATUS FOR EMPTY.        !!!!!!!
; !!!!! ALL WORKAROUND CODE IS CONDITIONAL AND THE EMULATOR CONSTANT MUST BE SET TO 1 IN     !!!!!!!
; !!!!! UART.INC          8/20/00 MNG								  !!!!!!!
;													  !!!!!!!
;													  !!!!!!!
 IF EMULATOR=1					;						  !!!!!!!
 IF USE_TIMER=1					;						  !!!!!!!
	OR	IMR,#00100000B			; Enable Timer 1 Interrupt 			  !!!!!!!
	LD	T1,#LOW(T1_BITTIMER)			; T1 value 					  !!!!!!!
	LD	PRE1,#(PRE1_BITTIMER<<2+11B)	; PRE1 value 					  !!!!!!!
 ENDIF							;						  !!!!!!!
 ENDIF							;						  !!!!!!!
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



  	EI
								
	RET		;All Done - ASCI ready to Send/Receive Data 

;***************************************************************************************************
;
SEND_BYTE:
	DI
	AND	UART_FLAGS,#(~TxRDY & ~RX_FLAG)	; The Tx is no longer ready (reset TxRDY)	
					      		; and we are in Tx mode (reset Rx_FLAG)

;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
; !!!!! THE E136 EMULATOR HAS A BUG THAT PREVENT Tx INTERRUPTS FROM THE ASCI FROM OCCURRING  !!!!!!!
; !!!!! A WORKAROUND USED HERE IS TO USE T1 AS A BIT TIME COUNTER, LET IT BE SERVICED BY THE !!!!!!
; !!!!! SAME SERVICE ROUTINE AS THE ASCI, AND USE IT TO TEST THE Tx STATUS FOR EMPTY.        !!!!!!!
; !!!!! ALL WORKAROUND CODE IS CONDITIONAL AND THE EMULATOR CONSTANT MUST BE SET TO 1 IN     !!!!!!!
; !!!!! UART.INC          8/20/00 MNG								  !!!!!!!
;													  !!!!!!!
;													  !!!!!!!
 IF EMULATOR=1			;								  !!!!!!!
 IF USE_TIMER=1			;								  !!!!!!!
	OR	TMR,#00001100B       ; Start the (Bit Width) timer				  !!!!!!!
 ENDIF					;								  !!!!!!!
 ENDIF					;								  !!!!!!!
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
													
	PUSH	RP										;Save RP before selecting new Register bank
	SRP	#%1A										;Select Uart bank

	LD		TDR,TxDATA								;Send byte

UART_EXIT:
	POP	RP
	EI
	RET



; As an alternative workaround to the emulator problem, the CHECK_XMIT routine has been provided.
; A user application can simply call CHECK_XMIT on a periodic basis instead of using the TMR1 to
; generate an interrupt to check the transmitter status.
;
; Use of this method is NOT recommended.  Only if both timers are needed by the application should
; this method be considered.

 IF USE_TIMER=0
;Check the transmit status and set/clear flags as needed
CHECK_XMIT:
	DI
	PUSH	RP
	SRP		#%1A									;Select uart bank
	TM		STAT,#TDRE				;Does the Tx need data to transfer?
	JR		Z,EXIT_CHECK									; No - Then service is done	
	OR		UART_FLAGS,#(TxRDY | UART_IRQ | RX_FLAG)	; Yes - Then set status flags for app to send more data
EXIT_CHECK

;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
; It has been observed that on bridge buffer overflows, that the HW UART Rx can get locked up as		!
; a result of re-initalizing the UART while data is still comming in.  The result is that the OE		!
; bit is set, but RDRNE is not.  To ensure that any Rx errors are always cleared the following code	!
; was added.  11/12/00 MNG												!
;															!
	TM	STAT,#(OE | PE | FE)		; Check for any Rx error conditions				!
	JR	Z,EXIT_CHECK1			; If no error occured just retun					!
;															!
; In order for the Error Flag Reset command to work, the Rx must be disabled FIRST!				!
;															!
	AND	CNTLA,#~RE		;Disable the Rx								!
	AND	CNTLA,#~EFR		;Clear the error flags in the HW						!
	OR	CNTLA,#RE		;Enable the Rx								!
;															!
	OR	UART_FLAGS,#UART_ERROR	;Set the error status bit for the application			!
;															!
EXIT_CHECK1														;
;															!
;															!
	POP	RP													;
	EI														;
	RET														;
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

 ENDIF


;**********************************************************************************************
;
; Hardware Interrupt Service Routine   HWUART_IRQ
;
;**********************************************************************************************

HWUART_IRQ:
	PUSH	RP
	SRP	#%1A				;Select uart bank
	TM	STAT,#RDRNE			;Does the Rx have data to transfer?
	JR	Z,TxDONE_SVC			; No - then this should be Tx Service

RxDONE_SVC
	TM	STAT,#(OE | PE | FE)		; Check for any Rx error conditions
	JR	NZ,RxERROR_SVC			; If an error occured - deal with it
	LD	RxDATA,RSR 				; no error - read byte from Uart Fifo
	OR	UART_FLAGS,#(RxRDY|UART_IRQ)	; Data available - set RxRDY and UART_IRQ flags
	JR	DONE_SVC				; xmit of BYTE is complete 

RxERROR_SVC
;															!
; In order for the Error Flag Reset command to work, the Rx must be disabled FIRST!				!
;															!
	AND	CNTLA,#~RE		;Disable the Rx								!
	AND	CNTLA,#~EFR		;Clear the error flags in the HW						!
	OR	CNTLA,#RE		;Enable the Rx								!
;
	OR	UART_FLAGS,#UART_ERROR	;Set the error status bit for the application
	LD	RxDATA,RSR			;Read the bad data just to 'pop' the UART FIFO (THIS MIGHT NOT BE NEEDED)???
	LD	RxDATA,#%FF			;Error will return FF data to the application (something must be there)
	JR	DONE_SVC


TxDONE_SVC
 IF USE_TIMER=1  ;THIS BLOCK OF CODE IS NORMALLY NEEDED, BUT NOT IF A MANUAL SAMPLE WORKAROUND IS USED
	TM	STAT,#TDRE			;Does the Tx need data to transfer?
	JR	Z,DONE_SVC					; No - Then service is done	
	OR	UART_FLAGS,#(TxRDY | UART_IRQ | RX_FLAG)	; Yes - Then set status flags for app to send more data
 ENDIF									;		but also return to the Rx mode until more data is 
									;		sent.

;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
; !!!!! THE E136 EMULATOR HAS A BUG THAT PREVENT Tx INTERRUPTS FROM THE ASCI FROM OCCURRING  !!!!!!!
; !!!!! A WORKAROUND USED HERE IS TO USE T1 AS A BIT TIME COUNTER, LET IT BE SERVICED BY THE !!!!!!!
; !!!!! SAME SERVICE ROUTINE AS THE ASCI, AND USE IT TO TEST THE Tx STATUS FOR EMPTY.        !!!!!!!
; !!!!! ALL WORKAROUND CODE IS CONDITIONAL AND THE EMULATOR CONSTANT MUST BE SET TO 1 IN     !!!!!!!
; !!!!! UART.INC          8/20/00 MNG								  !!!!!!!
;													  !!!!!!!
;													  !!!!!!!
 IF EMULATOR=1			;							  	  !!!!!!!
 IF USE_TIMER=1			;								  !!!!!!!
	AND	TMR,#11110011B	; Stop the timer if the Tx is done (it would keep running !!!!!!!
 ENDIF					; automatically)						  !!!!!!!
 ENDIF					;								  !!!!!!!
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

	
DONE_SVC											

;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
; It has been observed that on bridge buffer overflows, that the HW UART Rx can get locked up as		!
; a result of re-initalizing the UART while data is still comming in.  The result is that the OE		!
; bit is set, but RDRNE is not.  To ensure that any Rx errors are always cleared the following code	!
; was added.  11/12/00 MNG												!
;															!
	TM	STAT,#(OE | PE | FE)		; Check for any Rx error conditions				!
	JR	Z,DONE_SVC1			; If no error occured just retun					!
;															!
; In order for the Error Flag Reset command to work, the Rx must be disabled FIRST!				!
;															!
	AND	CNTLA,#~RE		;Disable the Rx								!
	AND	CNTLA,#~EFR		;Clear the error flags in the HW						!
	OR	CNTLA,#RE		;Enable the Rx								!
;															!
	OR	UART_FLAGS,#UART_ERROR	;Set the error status bit for the application			!
;															!
DONE_SVC1														;
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

	POP	RP
	IRET




 ELSE

;  IF UART_TYPE = 0 then SW UART has been selected and the following code is assembled

;**********************************************************************************************
;
;	                    SOFTWARE UART NOTES FOR Z8 MICROCONTROLLERS
;
;  This program will provide an interrupt driven HALF DUPLEX software UART for the Z8 family
;  of microcontrollers. It will not support simultaneous activity on the Tx & Rx pins.  Rx 
;  will take priority over Tx, causing a Tx to be suspended and restarted at a later time if
;  necessary. (NOT YET IMPLEMENTED)
;
;  The UART has been tested at 1200, 2400, 4800, 9600 and 19200 Baud using a 12mhz crystal
;  and has been found to be reliable at all rates.  The user is cautioned to allow sufficent
;  interrupt service time (and low service latency) in the application code if operation at
;  the higher baud rates is desired. 
;
;  System Resources Used:
;	147 Bytes ROM	(182 Bytes ROM if UART_SHARING selected)
;	5 Bytes RAM
;	1 C/T (T1)
;	2 I/O pins (P20 & P31) Note: the input pin must have edge interrupt capability 
;
;  DO NOT EDIT THIS SOURCE FILE.  
;  ALL APPLICATION CUSTOMIZATION IS DONE IN THE UART.INC FILE
;

;**********************************************************************************************
; 	Initialization      	INIT_UART is a GLOBAL ENTRY POINT                                          
;**********************************************************************************************				
INIT_UART:              ;ENTRY POINT to initialize the SW UART
	OR	IMR,#00100001B			; Enable Interrupts that are needed (RX input and Timer)
	LD	P3M,#00000001B			; P30-P33=>digital mode, P2 Push-Pull
	OR	P0,#TxD_Bit				; Set TxD bit high (Quiescent)
	LD	T1,#LOW(T1_BITTIMER)			; T1 value for the specified Baud Rate
	LD	PRE1,#(PRE1_BITTIMER<<2+11B)	; PRE1 value for the specified Baud Rate

; The Tx is ready for data as soon as intialization is completed, so TxRDY gets set now
; The Rx is enabled as soon as intialization is completed, so any inbound messages will be rec'd.

	LD	UART_FLAGS,#00000110B	; Initial state of Flags:
					;	RxRDY     = 0 - There is no Rx data
					;	TxRDY     = 1 - Ready for Tx data
					;	RX_FLG    = 1 - We'll be set for Rx after init
					;	UART_IRQ  = 0 - There is no transfer completed
					;  UART_BUSY = 0 - The UART is not running (BUSY)
					;  UART_DLY  = 0 - The UART has not just ended an Rx
;**********************************************************************************************
; 	Enable RX   		EN_RCV is a GLOBAL ENTRY POINT
;**********************************************************************************************
EN_RCV:                       ; ENTRY POINT to allow SW UART to listen for new RxD
	OR	UART_FLAGS,#RX_FLAG		; Since we are Rx mode, set the Rx_FLAG
	AND	UART_FLAGS,#~RxRDY    	; Clear the RxRDY flag (no data avail yet.)
	OR	IMR,#00010001B		; Enable T0, and P32 (F) ext interrupts 
	RET                        	; All done - reception will be handled by interrupts
	
	
;**********************************************************************************************
;   	Transmit Byte Routine 	SEND_BYTE is a GLOBAL ENTRY POINT
;********************************************************************************************** 

SEND_BYTE:                 		; ENTRY POINT to start new byte transmission
	DI
	AND	IMR,#11111110B                 	; Disable the Rx edge detect interrupt (HDX operation)
	AND	UART_FLAGS,#(~TxRDY & ~RX_FLAG)	; The Tx is no longer ready (reset TxRDY)	
					      			; and we are in Tx mode (reset Rx_FLAG)
 IF UART_SHARING = 1
 	OR	UART_FLAGS,#UART_BUSY		; If UART SHARING - Set the UART_BUSY indication
	AND	UART_FLAGS,#~UART_DLY		;       and clear the UART_DLY if it was running
 ENDIF

	LD	BitCount,#10				; Start Bit + 8 data + 1 Stop = 10 bit count
	AND	P0,#~TxD_Bit                  	; Set the TxD bit low for Start bit generation
	OR	TMR,#00001100B          		; Start the (Bit Width) timer
	EI
	RET                                   	; All done - xmission will be handled by interrupts
							;            for the rest of this byte
;**********************************************************************************************
; 	FALLING EDGE Interrupt	START_RX is a GLOBAL INTERRUPT SVC ENTRY POINT  	
;**********************************************************************************************				
START_RX:
	TM	P3,#RxD_Bit			; Check RxD should be low if we are receiving a START Bit
	JR	NZ, NOT_RECEIVING     	; If no START bit (maybe glitch) not really receiving
	LD	T1,#LOW T1_H_BITTIMER   	; set up bit-time at 1/2 bit for center sampling
	OR	TMR,#00001100B		; start Timer
	LD	T1,#LOW T1_BITTIMER		; set up bit-time for remaining bits 
	LD	BitCount,#10          	; expecting 10 bits in transmission
	OR	UART_FLAGS,#RX_FLAG   	; THIS MAY BE UNECESSARY (was done in EN_REC)
							;    or it might be required if SEND_BYTE is called just as
							;    an Rx interrupt comes in - TO BE CHECKED OUT FURTHER

 IF UART_SHARING = 1
 	OR	UART_FLAGS,#UART_BUSY	; If UART SHARING - Set the UART_BUSY indication
	AND	UART_FLAGS,#~UART_DLY	;       and clear the UART_DLY if it was running
 ENDIF
					
	AND	IMR,#11111110B		; Disable IRQ0 F-edge interrupt

NOT_RECEIVING:
	IRET
	
;**********************************************************************************************
; 	TIMER1 Interrupt	IRQ5_T1 is a GLOBAL INTERRUPT SVC ENTRY POINT
;*******************************************************************************
IRQ5_T1:					
	TM	UART_FLAGS,#RX_FLAG		; Check if we are to RX or TX 
	JR	Z, TRANSMIT


;;*************TEST PATCH FOR CALIBRATING BIT SAMPLE POSITION (T1 value)***********************
;;          
;;           Used for bit sample point measurements with scope
;;
;;	AND	P0,#11111101B           ;Put a pulse on the TxD pin
;;	OR	P0,#00000010B		
;;
;;
;;***********************************END OF TEST PATCH*****************************************
	


;**********************************************************************************************
;	Receiving 
;**********************************************************************************************

 IF UART_SHARING = 1
 	TM	UART_FLAGS,#UART_DLY				; If UART SHARING - See if we are completing an Rx delay
	JR	Z,RECEIVE
 	AND	UART_FLAGS,#(~UART_BUSY & ~UART_DLY)	; cycle - if so clear the UART_BUSY & DELAY indication
	AND	TMR,#11110011B					; Stop the timer
	IRET								; and return
 ENDIF

RECEIVE:	
	CP	BitCount, #10		; If the bitcount is still 10 then this is the START BIT
	JR	NE, NOT_START           ; Otherwise its a data bit

START_BIT:	                        ; Here to process the START BIT
	TM	P3, #RxD_Bit		; check if RxD is 0 (Start bit should be = 0)
	JR	NZ,STOP_UART		; If not, stop the UART (glitch?)
	DEC	BitCount                ; If so, decrement the bit count
	JR	EXIT_UART              	; and wait for the next data bit
;----------------------------------------------------------------------------------------------			
NOT_START:
      	DEC	BitCount               	; Decrement the bit count
      	JR	Z,STOP_BIT             	; If zero then this is the STOP BIT
      					; If not zero then this is a data bit
DATA_BIT:
	RCF                             ; RESET CARRY
	TM	P3,#RxD_Bit		; Check if received bit is 0 or 1
	JR	Z,SKIP
	SCF				; If 1 set Carry flag as data bit
SKIP:
	RRC	RCV_DATA_BUFF		; Rotate data bit
	JR	EXIT_UART	       	; Exit the service and wait for the next bit
;----------------------------------------------------------------------------------------------

	
STOP_BIT:                      	; The STOP BIT should be High
	TM	P3,#RxD_Bit             ; Is the Rx Pin Low (BREAK)?
	JR	Z,STOP_UART             ;     If so - dump the data and exit
	LD	RxDATA, RCV_DATA_BUFF   ;     If not - valid STOP, save data

 	OR	UART_FLAGS,#(RxRDY|UART_IRQ)	; Data available (set RxRDY) and
							; xmit of BYTE is complete (set UART_IRQ)

 IF UART_SHARING = 1
 	OR	UART_FLAGS,#UART_DLY			; If UART SHARING - Set the UART_DLY indication
 ENDIF							; to generate a 1 bit time delay before clearing BUSY

	JR	STOP_UART

;**********************************************************************************************
;	Transmitting  
;**********************************************************************************************
;
; As we serialize the byte to be transmitted, we fill the 'empty' bit positions with 1
; (from the carry).  This will later be used for the stop bit(s)
;	
TRANSMIT:
	SCF				; Set carry flag=1
	RRC	TxDATA			; Rotate with carry
	JR	C,SEND_ONE		; If carry set, bit must be 1
     	AND	P0,#~TxD_Bit		; Send zero bit on port P20
     	JR	NEXT_BIT
;----------------------------------------------------------------------------------------------	
SEND_ONE:
	OR	P0,#TxD_Bit		; Send one bit on port 20 pin1

NEXT_BIT:
	DEC 	BitCount              ; Decrement the bit counter
	JR	NZ,EXIT_UART		; If more bits to serialize, exit the service
DONE:	
	OR	UART_FLAGS,#(TxRDY |UART_IRQ)	; Transmission of BYTE is complete
							; and set 'Interrupt' bit

 IF UART_SHARING = 1
 	AND	UART_FLAGS,#~UART_BUSY		; If UART SHARING - Clear the UART_BUSY indication
 ENDIF

			
STOP_UART:
 IF UART_SHARING = 1
 	TM	UART_FLAGS,#UART_DLY		; If UART SHARING - Don't stop the C/T if DLY is undeway
	JR	NZ,STOP_UART_1
 ENDIF

	AND	TMR,#11110011B		; Stop the timer

STOP_UART_1:
	OR	UART_FLAGS,#RX_FLAG	; Turn on Receive mode again
	OR	IMR,#00000001B	; Enable P32
	
EXIT_UART:	         
	IRET


	ENDIF
